<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\CompoundHyip;
use App\Models\AdminNotification;
use App\Models\Contribution;
use App\Models\Investment;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\Rule;

class InvestmentController extends Controller
{
    public function investInsert(Request $request)
    {
        $request->validate([
            'initial_deposit'        => 'required|numeric|gt:0',
            'contribution_amount'    => 'required|numeric|gt:0',
            'contribution_frequency' => ['required', Rule::in([Status::DAILY_FREQUENCY, Status::MONTHLY_FREQUENCY, Status::ANNUALLY_FREQUENCY, Status::WEEKLY_FREQUENCY])],
            'duration_years'         => 'required|integer|min:' . gs('investment_minimum_year') . '|max:' . gs('investment_maximum_year'),
            'payment_method'         => ['required', Rule::in([Status::WALLET_INVEST, Status::GATEWAY_INVEST])],
        ]);

        $user = auth()->user();

        if ($request->payment_method == Status::WALLET_INVEST && $user->balance < $request->initial_deposit) {
            $notify[] = ['error', 'Insufficient balance. Please deposit funds to your wallet.'];
            return back()->withNotify($notify);
        }

        $investment                         = new Investment();
        $investment->user_id                = $user->id;
        $investment->initial_deposit        = $request->initial_deposit;
        $investment->contribution_amount    = $request->contribution_amount;
        $investment->contribution_frequency = $request->contribution_frequency;

        switch ($request->contribution_frequency) {
            case Status::DAILY_FREQUENCY:
                $investment->next_contribution_date = now()->addDay();
                break;
            case Status::WEEKLY_FREQUENCY:
                $investment->next_contribution_date = now()->addWeek();
                break;
            case Status::MONTHLY_FREQUENCY:
                $investment->next_contribution_date = now()->addMonth();
                break;
            case Status::ANNUALLY_FREQUENCY:
                $investment->next_contribution_date = now()->addYear();
                break;
        }

        $investment->duration_years     = $request->duration_years;
        $investment->annual_return_rate = gs('annual_return_rate');
        $investment->start_date         = now();
        $investment->end_date           = now()->addYears((int) $request->duration_years);
        $investment->payment_status     = Status::PAYMENT_INITIATE;

        $investmentReturn          = $this->calculateInvestmentReturn($investment);
        $investment->total_return  = $investmentReturn;
        $investment->return_amount = $investment->initial_deposit;

        $contributionNumber             = $this->getContributionFrequencyFactor($investment->contribution_frequency);
        $investment->contribution_count = $contributionNumber * $investment->duration_years;
        $investment->save();

        if ($request->payment_method == Status::WALLET_INVEST) {
            if ($user->balance < $investment->initial_deposit) {
                $notify[] = ['error', 'Insufficient balance. Please deposit funds to your wallet.'];
                return back()->withNotify($notify);
            }

            self::updateInvestedData($investment);

            $notify[] = ['success', 'Congratulations, investment successful'];
            return to_route('user.investment.all')->withNotify($notify);
        } else {
            session(['investment_id' => $investment->id]);
            return to_route('user.deposit.index');
        }
    }

    public static function updateInvestedData($investment, $gateway = null)
    {
        $user = $investment->user;
        if ($user->balance < $investment->initial_deposit) {
            return;
        }
        $user->balance -= $investment->initial_deposit;
        $user->save();

        if ($gateway) {
            $details = "Investment via " . $gateway;
        } else {
            $details = "Investment via wallet";
        }

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $investment->initial_deposit;
        $transaction->post_balance = $user->balance;
        $transaction->charge       = 0;
        $transaction->trx_type     = '-';
        $transaction->remark       = 'investment_deposit';
        $transaction->details      = $details;
        $transaction->trx          = getTrx();
        $transaction->save();

        $investment->payment_status = Status::PAYMENT_SUCCESS;
        $investment->trx            = $transaction->trx;
        $investment->status         = Status::INVESTMENT_ACTIVE;
        $investment->payment_method = $gateway ? Status::GATEWAY_INVEST : Status::WALLET_INVEST;
        $investment->save();

        if (gs('invest_commission')) {
            $commissionType = 'invest_commission';
            CompoundHyip::levelCommission($user, $investment->initial_deposit, $commissionType, $transaction->trx, gs());
        }

        $frequencyText = frequencyText($investment->contribution_frequency);

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = 'New investment from ' . $user->username;
        $adminNotification->click_url = urlPath('admin.investment.details', $investment->id);
        $adminNotification->save();

        notify($user, 'INVESTMENT_SUCCESSFUL', [
            'initial_deposit'        => showAmount($investment->initial_deposit, currencyFormat: false),
            'contribution_amount'    => showAmount($investment->contribution_amount, currencyFormat: false),
            'contribution_frequency' => $frequencyText,
            'investment_start_date'  => showDateTime($investment->created_at),
            'investment_end_date'    => showDateTime($investment->end_date),
            'trx'                    => $transaction->trx,
            'post_balance'           => showAmount($user->balance, currencyFormat: false),
        ]);
        return true;
    }

    public function calculateInvestmentReturn($investment)
    {
        $initialDeposit     = $investment->initial_deposit;
        $contributionAmount = $investment->contribution_amount;
        $returnRate         = $investment->annual_return_rate / 100;
        $contributionNumber = $this->getContributionFrequencyFactor($investment->contribution_frequency);
        $duration           = $investment->duration_years;

        $finalInvest        = $initialDeposit * pow((1 + $returnRate / $contributionNumber), $contributionNumber * $duration);
        $finalInterest      = $contributionAmount * ((pow(1 + $returnRate / $contributionNumber, $contributionNumber * $duration) - 1) / ($returnRate / $contributionNumber));
        $totalFutureBalance = $finalInvest + $finalInterest;

        return $totalFutureBalance;
    }

    private function getContributionFrequencyFactor($frequency)
    {
        switch ($frequency) {
            case Status::DAILY_FREQUENCY:
                return 365;
            case Status::WEEKLY_FREQUENCY:
                return 52;
            case Status::MONTHLY_FREQUENCY:
                return 12;
            case Status::ANNUALLY_FREQUENCY:
                return 1;
        }
    }

    public function allInvestments($scope = null)
    {
        $pageTitle = $scope ? ucfirst($scope) . ' Investments' : 'All Investments';
        $user      = auth()->user();

        $query = Investment::where('user_id', $user->id)->searchable(['trx']);

        if ($scope) {
            $query = $query->{$scope}();
        }

        $investments = $query->orderBy('id', 'desc')->paginate(getPaginate());

        return view('Template::user.investment.index', compact('pageTitle', 'investments', 'scope'));
    }

    public function details($investmentId)
    {
        $pageTitle          = 'Investment Details';
        $user               = auth()->user();
        $contributions      = Contribution::with('investment')->where('user_id', $user->id)->where('investment_id', $investmentId)->searchable(['trx'])->orderBy('id', 'desc')->paginate(getPaginate());
        $paidContributions  = Contribution::where('user_id', $user->id)->where('investment_id', $investmentId)->where('status', Status::CONTRIBUTION_PAID)->get();
        $contributionAmount = $paidContributions->sum('amount');
        $investment         = Investment::where('id', $investmentId)->where('user_id', $user->id)->firstOrFail();
        return view('Template::user.investment.details', compact('pageTitle', 'investment', 'contributions', 'contributionAmount'));
    }

    public function getContributions()
    {
        $pageTitle = 'All Contributions';
        $user      = auth()->user();

        $contributions = Contribution::with('investment')
            ->where('user_id', $user->id)
            ->searchable(['trx'])
            ->orderBy('id', 'desc')
            ->paginate(getPaginate());

        return view('Template::user.investment.contribution', compact('pageTitle', 'contributions'));
    }

    public function closeInvestment($id)
    {
        $user = auth()->user();

        if (gs('investment_early_closure') == Status::INVESTMENT_CLOSURE_DISABLE) {
            $notify[] = ['error', 'Investment close request is currently disabled by the administration'];
            return back()->withNotify($notify);
        }
        $investment = Investment::active()->where('user_id', $user->id)->where('id', $id)->firstOrFail();
        if ($investment->close_request == Status::YES) {
            $notify[] = ['error', 'You have already sent a close request for this investment. Please wait for approval.'];
            return back()->withNotify($notify);
        }

        $investment->close_request        = Status::YES;
        $investment->close_request_status = Status::CLOSE_STATUS_PENDING;
        $investment->save();

        $notify[] = ['success', 'The investment close request has been sent. Please wait for approval.'];
        return back()->withNotify($notify);
    }

    public function investNow()
    {
        $pageTitle = 'Invest Now';
        return view('Template::user.investment.invest_now', compact('pageTitle'));
    }
}
