<?php

namespace App\Http\Controllers;

use App\Constants\Status;
use App\Lib\CompoundHyip;
use App\Lib\CurlRequest;
use App\Models\AdminNotification;
use App\Models\Contribution;
use App\Models\CronJob;
use App\Models\CronJobLog;
use App\Models\Investment;
use App\Models\Transaction;
use Carbon\Carbon;

class CronController extends Controller {
    public function cron() {
        $general            = gs();
        $general->last_cron = now();
        $general->save();

        $crons = CronJob::with('schedule');

        if (request()->alias) {
            $crons->where('alias', request()->alias);
        } else {
            $crons->where('next_run', '<', now())->where('is_running', Status::YES);
        }
        $crons = $crons->get();
        foreach ($crons as $cron) {
            $cronLog              = new CronJobLog();
            $cronLog->cron_job_id = $cron->id;
            $cronLog->start_at    = now();
            if ($cron->is_default) {
                $controller = new $cron->action[0];
                try {
                    $method = $cron->action[1];
                    $controller->$method();
                } catch (\Exception $e) {
                    $cronLog->error = $e->getMessage();
                }
            } else {
                try {
                    CurlRequest::curlContent($cron->url);
                } catch (\Exception $e) {
                    $cronLog->error = $e->getMessage();
                }
            }
            $cron->last_run = now();
            $cron->next_run = now()->addSeconds($cron->schedule->interval);
            $cron->save();

            $cronLog->end_at = $cron->last_run;

            $startTime         = Carbon::parse($cronLog->start_at);
            $endTime           = Carbon::parse($cronLog->end_at);
            $diffInSeconds     = $startTime->diffInSeconds($endTime);
            $cronLog->duration = $diffInSeconds;
            $cronLog->save();
        }
        if (request()->target == 'all') {
            $notify[] = ['success', 'Cron executed successfully'];
            return back()->withNotify($notify);
        }
        if (request()->alias) {
            $notify[] = ['success', keyToTitle(request()->alias) . ' executed successfully'];
            return back()->withNotify($notify);
        }
    }

    public function notifyUpcomingContributions() {
        $tomorrow    = Carbon::now()->addDay()->startOfDay();
        $investments = Investment::active()->whereDate('next_contribution_date', $tomorrow)->where('end_date', '>', Carbon::now())->get();
        foreach ($investments as $investment) {
            $user = $investment->user;
            notify($user, 'UPCOMING_CONTRIBUTION_REMINDER', [
                'contribution_amount'    => showAmount($investment->contribution_amount, currencyFormat: false),
                'next_contribution_date' => showDateTime($investment->next_contribution_date),
                'balance'                => showAmount($user->balance, currencyFormat: false),
            ]);
        }
    }

    public function processContributions() {

        $investments = Investment::active()->where('next_contribution_date', '<=', Carbon::now())->whereColumn('paid_contribution', '<', 'contribution_count')->where('end_date', '>', Carbon::now())->limit(50)->get();

        foreach ($investments as $investment) {

            $user = $investment->user;

            $contribution                = new Contribution();
            $contribution->investment_id = $investment->id;
            $contribution->user_id       = $user->id;
            $contribution->amount        = $investment->contribution_amount;
            $contribution->save();

            if ($user->balance >= $investment->contribution_amount) {

                $user->balance -= $investment->contribution_amount;
                $user->save();

                $transaction               = new Transaction();
                $transaction->user_id      = $user->id;
                $transaction->amount       = $investment->contribution_amount;
                $transaction->post_balance = $user->balance;
                $transaction->charge       = 0;
                $transaction->trx_type     = '-';
                $transaction->details      = getAmount($investment->contribution_amount) . ' ' . gs('cur_text') . ' deduct for contribution amount';
                $transaction->trx          = getTrx();
                $transaction->remark       = 'contribution_amount';
                $transaction->save();

                $contribution->status           = Status::CONTRIBUTION_PAID;
                $contribution->contributed_date = Carbon::now();
                $contribution->trx              = $transaction->trx;
                $contribution->save();

                $investment->next_contribution_date = $this->calculateNextContributionDate($investment->contribution_frequency);

                $this->updateReturnAmount($investment);

                notify($user, 'CONTRIBUTION_DEDUCTED', [
                    'contribution_amount' => showAmount($investment->contribution_amount, currencyFormat: false),
                    'contribution_date'   => showDateTime($contribution->contributed_date),
                    'trx'                 => $transaction->trx,
                    'post_balance'        => showAmount($user->balance, currencyFormat: false),
                ]);
            } else {
                $penaltyRate = getAmount(gs('contribution_penalty'));
                $penaltyType = gs('penalty_type');

                $penaltyAmount = ($penaltyType == Status::PENALTY_TYPE_PERCENT) ? ($contribution->amount * $penaltyRate) / 100 : $penaltyRate;

                $contribution->penalty_amount += $penaltyAmount;
                $contribution->status = Status::CONTRIBUTION_UNPAID;
                $contribution->save();

                $investment->next_contribution_date = $this->calculateNextContributionDate($investment->contribution_frequency);
                $investment->save();

                notify($user, 'CONTRIBUTION_FAILED', [
                    'contribution_amount' => showAmount($investment->contribution_amount, currencyFormat: false),
                    'balance'             => showAmount($user->balance, currencyFormat: false),
                    'penalty_amount'      => showAmount($penaltyAmount, currencyFormat: false),
                ]);
            }
        }
    }

    public function dueContributions() {
        $contributions = Contribution::unpaid()->withWhereHas('investment', function ($query) {
            $query->active()->where('end_date', '>', Carbon::now())
                ->whereHas('user');
        })->get();

        foreach ($contributions as $contribution) {

            $investment = $contribution->investment;
            $user       = $investment->user;

            $contributionAmount = $contribution->amount + $contribution->penalty_amount;

            if ($user->balance >= $contributionAmount) {
                $user->balance -= $contributionAmount;
                $user->save();

                $transaction               = new Transaction();
                $transaction->user_id      = $user->id;
                $transaction->amount       = $contributionAmount;
                $transaction->post_balance = $user->balance;
                $transaction->charge       = 0;
                $transaction->trx_type     = '-';
                $transaction->details      = getAmount($contributionAmount) . ' ' . gs('cur_text') . ' deduct for contribution due';
                $transaction->trx          = getTrx();
                $transaction->remark       = 'contribution_due';
                $transaction->save();

                $contribution->status           = Status::CONTRIBUTION_PAID;
                $contribution->contributed_date = Carbon::now();
                $contribution->trx              = $transaction->trx;
                $contribution->save();

                $this->updateReturnAmount($investment);

                notify($user, 'DUE_CONTRIBUTION', [
                    'contribution_amount' => showAmount($contribution->amount, currencyFormat: false),
                    'penalty_amount'      => showAmount($contribution->penalty_amount, currencyFormat: false),
                    'total_amount'        => showAmount($contributionAmount, currencyFormat: false),
                    'contribution_date'   => showDateTime($contribution->contributed_date),
                    'trx'                 => $transaction->trx,
                    'post_balance'        => showAmount($user->balance, currencyFormat: false),
                ]);
            }
        }
    }

    public function updateReturnAmount($investment) {
        $investment->paid_contribution += 1;

        $initialDeposit     = $investment->initial_deposit;
        $contributionAmount = $investment->contribution_amount;
        $returnRate         = $investment->annual_return_rate / 100;
        $contributionNumber = $this->getInterestFrequencyFactor($investment->contribution_frequency);

        $duration           = $investment->paid_contribution / $contributionNumber;
        $finalInvest        = $initialDeposit * pow((1 + $returnRate / $contributionNumber), $contributionNumber * $duration);
        $finalInterest      = $contributionAmount * ((pow(1 + $returnRate / $contributionNumber, $contributionNumber * $duration) - 1) / ($returnRate / $contributionNumber));
        $totalFutureBalance = $finalInvest + $finalInterest;

        $investment->return_amount = $totalFutureBalance;
        $investment->save();
    }

    private function getInterestFrequencyFactor($frequency) {
        switch ($frequency) {
        case Status::DAILY_FREQUENCY:
            return 365;
        case Status::WEEKLY_FREQUENCY:
            return 52;
        case Status::MONTHLY_FREQUENCY:
            return 12;
        case Status::ANNUALLY_FREQUENCY:
            return 1;
        }
    }

    public function applyPenaltyIncrease() {
        $penaltyRate      = getAmount(gs('contribution_penalty'));
        $penaltyType      = gs('penalty_type');
        $penaltyFrequency = gs('penalty_frequency');

        $contributions = Contribution::unpaid()->get();
        foreach ($contributions as $contribution) {
            if (!$this->shouldApplyPenalty($contribution->updated_at, $penaltyFrequency)) {
                continue;
            }
            $dailyPenalty = ($penaltyType == Status::PENALTY_TYPE_PERCENT) ? ($contribution->amount * $penaltyRate) / 100 : $penaltyRate;
            $contribution->penalty_amount += $dailyPenalty;
            $contribution->save();
        }
    }

    private function shouldApplyPenalty($lastUpdated, $frequency): bool {
        $now = now();
        switch ($frequency) {
        case Status::DAILY_FREQUENCY:
            return $lastUpdated->diffInDays($now) >= 1;
        case Status::WEEKLY_FREQUENCY:
            return $lastUpdated->diffInWeeks($now) >= 1;
        case Status::MONTHLY_FREQUENCY:
            return $lastUpdated->diffInMonths($now) >= 1;
        case Status::ANNUALLY_FREQUENCY:
            return $lastUpdated->diffInYears($now) >= 1;
        default:
            return false;
        }
    }

    private function calculateNextContributionDate($frequency) {
        switch ($frequency) {
        case Status::DAILY_FREQUENCY:
            return Carbon::now()->addDay();
        case Status::WEEKLY_FREQUENCY:
            return Carbon::now()->addWeek();
        case Status::MONTHLY_FREQUENCY:
            return Carbon::now()->addMonth();
        case Status::ANNUALLY_FREQUENCY:
            return Carbon::now()->addYear();
        }
    }

    public function processCompletedInvestments() {
        $investments = Investment::active()->where('end_date', '<=', Carbon::now())->with('contributions')->limit(30)->get();
        foreach ($investments as $investment) {

            $unpaidContributions = $investment->contributions()->where('status', Status::CONTRIBUTION_UNPAID)->exists();

            if (!$unpaidContributions) {
                $user = $investment->user;
                $user->balance += $investment->total_return;
                $user->save();

                $transaction               = new Transaction();
                $transaction->user_id      = $user->id;
                $transaction->amount       = $investment->total_return;
                $transaction->post_balance = $user->balance;
                $transaction->charge       = 0;
                $transaction->trx_type     = '+';
                $transaction->details      = 'Investment return successfully to your account';
                $transaction->trx          = getTrx();
                $transaction->remark       = 'investment_return';
                $transaction->save();

                $investment->status = Status::INVESTMENT_COMPLETE;
                $investment->save();

                // Referral Commission if Enabled
                if (gs('invest_return_commission')) {
                    $commissionType = 'invest_return_commission';
                    CompoundHyip::levelCommission($user, $investment->total_return, $commissionType, $transaction->trx, gs());
                }

                $frequencyText = frequencyText($investment->contribution_frequency);

                $adminNotification            = new AdminNotification();
                $adminNotification->user_id   = $user->id;
                $adminNotification->title     = 'Investment Return Successful to ' . $user->username;
                $adminNotification->click_url = urlPath('admin.investment.details', $investment->id);
                $adminNotification->save();

                notify($user, 'INVESTMENT_RETURN_SUCCESSFUL', [
                    'initial_deposit'        => showAmount($investment->initial_deposit, currencyFormat: false),
                    'contribution_amount'    => showAmount($investment->contribution_amount, currencyFormat: false),
                    'contribution_frequency' => $frequencyText,
                    'total_return'           => showAmount($investment->total_return, currencyFormat: false),
                    'investment_end_date'    => showDateTime($investment->end_date),
                    'trx'                    => $transaction->trx,
                    'post_balance'           => showAmount($user->balance, currencyFormat: false),
                ]);
            }
        }
    }
}
