<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Contribution;
use App\Models\Investment;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class InvestmentController extends Controller
{
    public function listInvestments($scope = null, $user = null)
    {
        $investmentsQuery = Investment::with('user')
            ->orderBy('id', 'desc')
            ->searchable(['trx', 'user:username'])
            ->dateFilter();

        if ($scope) {
            $investmentsQuery = $investmentsQuery->{$scope}();
        }

        if (!is_null($user)) {
            $investmentsQuery->where('user_id', $user);
        }

        $investments = $investmentsQuery->paginate(getPaginate());

        $pageTitle = ($scope ? ucfirst($scope) : 'All') . ' Investments';

        return view('admin.investment.list', compact('pageTitle', 'investments'));
    }

    public function all($user = null)
    {
        return $this->listInvestments(null, $user);
    }

    public function active($user = null)
    {
        return $this->listInvestments('active', $user);
    }

    public function completed($user = null)
    {
        return $this->listInvestments('completed', $user);
    }

    public function closed($user = null)
    {
        return $this->listInvestments('closed', $user);
    }

    public function details($id)
    {
        $pageTitle     = 'Investment Details';
        $investment    = Investment::findOrFail($id);
        $contributions = Contribution::with('investment')
            ->where('investment_id', $id)
            ->orderBy('id', 'desc')
            ->searchable(['trx'])
            ->dateFilter()
            ->paginate(getPaginate());
        return view('admin.investment.details', compact('pageTitle', 'investment', 'contributions'));
    }

    public function closeRequests($user = null)
    {
        $investmentsQuery = Investment::active()
            ->closeRequest()
            ->orderBy('id', 'desc')
            ->with('user')
            ->searchable(['trx', 'user:username'])
            ->dateFilter();

        if (!is_null($user)) {
            $investmentsQuery->where('user_id', $user);
        }

        $investments = $investmentsQuery->paginate(getPaginate());
        $pageTitle   = 'Investment Close Requests';
        return view('admin.investment.close_requests', compact('pageTitle', 'investments'));
    }

    public function closeDetails($id)
    {
        $pageTitle     = 'Investment Close Request';
        $contributions = Contribution::with('investment')
            ->where('investment_id', $id)
            ->orderBy('id', 'desc')
            ->searchable(['trx'])
            ->dateFilter()
            ->paginate(getPaginate());
        $investment = Investment::closeRequest()->findOrFail($id);
        return view('admin.investment.close_details', compact('pageTitle', 'investment', 'contributions'));
    }

    public function closeRequestApprove($id)
    {
        if (!gs('investment_early_closure')) {
            $notify[] = ['error' => 'Early closure is disabled'];
            return back()->withNotify($notify);
        }
        $investment = Investment::active()->where('close_request', Status::YES)->findOrFail($id);
        if (!$investment) {
            $notify[] = ['error' => 'Investment not found'];
            return back()->withNotify($notify);
        }
        $user = $investment->user;
        if (!$user) {
            $notify[] = ['error' => 'User not found'];
            return back()->withNotify($notify);
        }

        $return = $this->calculateEarlyClosureReturn($investment);

        $user->balance += $return;
        $user->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $return;
        $transaction->post_balance = $user->balance;
        $transaction->trx_type     = '+';
        $transaction->details      = 'Early closure return';
        $transaction->trx          = getTrx();
        $transaction->remark       = 'early_closure_return';
        $transaction->save();

        $investment->status               = Status::INVESTMENT_CLOSED;
        $investment->close_request        = Status::NO;
        $investment->close_request_status = Status::CLOSE_STATUS_APPROVED;
        $investment->save();

        $frequencyText = frequencyText($investment->contribution_frequency);

        notify($user, 'INVESTMENT_CLOSE_REQUEST_APPROVED', [
            'initial_deposit'        => showAmount($investment->initial_deposit, currencyFormat: false),
            'contribution_amount'    => showAmount($investment->contribution_amount, currencyFormat: false),
            'investment_start_date'  => showDateTime($investment->created_at),
            'investment_end_date'    => showDateTime($investment->end_date),
            'contribution_frequency' => $frequencyText,
            'annual_return_rate'     => getAmount($investment->annual_return_rate),
            'total_return'           => showAmount($investment->total_return, currencyFormat: false),
            'duration_years'         => $investment->duration_years,
            'closure_return'         => showAmount($return, currencyFormat: false),
            'post_balance'           => showAmount($user->balance, currencyFormat: false),
            'trx'                    => $transaction->trx,
            'closure_date'           => showDateTime(now()),
        ]);

        $notify[] = ['success', 'Investment closed successfully, amount including profit credited.'];
        return to_route('admin.investment.requests.close')->withNotify($notify);
    }

    public function calculateEarlyClosureReturn($investment)
    {

        $returnAmount = getAmount($investment->return_amount);

        $penaltyAmount = 0;

        if (gs('early_closure_penalty') > 0) {
            $penaltyPercentage = getAmount(gs('early_closure_penalty')) / 100;
            $penaltyAmount     = $returnAmount * $penaltyPercentage;
        }

        $finalReturn = ($returnAmount - $penaltyAmount);
        return $finalReturn;
    }

    public function closeRequestReject($id)
    {
        $investment = Investment::findOrFail($id);
        if (!$investment) {
            $notify[] = ['error' => 'Investment not found'];
            return back()->withNotify($notify);
        }
        $investment->close_request        = Status::NO;
        $investment->close_request_status = Status::CLOSE_STATUS_REJECTED;
        $investment->save();

        $frequencyText = frequencyText($investment->contribution_frequency);

        notify($investment->user, 'INVESTMENT_CLOSE_REQUEST_REJECTED', [
            'initial_deposit'        => showAmount($investment->initial_deposit, currencyFormat: false),
            'contribution_amount'    => showAmount($investment->contribution_amount, currencyFormat: false),
            'investment_start_date'  => showDateTime($investment->created_at),
            'investment_end_date'    => showDateTime($investment->end_date),
            'contribution_frequency' => $frequencyText,
            'trx'                    => $investment->trx,
            'annual_return_rate'     => $investment->annual_return_rate,
            'total_return'           => showAmount($investment->total_return, currencyFormat: false),
            'duration_years'         => $investment->duration_years,
        ]);

        $notify[] = ['success', 'Investment close request has been rejected'];
        return to_route('admin.investment.requests.close')->withNotify($notify);
    }

    public function setting()
    {
        $pageTitle = 'Investment Setting';
        return view('admin.investment.setting', compact('pageTitle'));
    }

    public function settingUpdate(Request $request) {
        $request->validate([
            'annual_return_rate'      => 'required|numeric|min:0|max:100',
            'investment_minimum_year' => 'required|integer|min:1|lt:investment_maximum_year',
            'investment_maximum_year' => 'required|integer|min:1|gt:investment_minimum_year',
            'penalty_type'            => ['required', Rule::in([Status::PENALTY_TYPE_FIXED, Status::PENALTY_TYPE_PERCENT])],
            'contribution_penalty'    => 'required|numeric',
            'early_closure_penalty'   => 'required|numeric|min:0|max:100',
            'penalty_frequency'       => ['required', Rule::in([Status::DAILY_FREQUENCY, Status::WEEKLY_FREQUENCY, Status::MONTHLY_FREQUENCY, Status::ANNUALLY_FREQUENCY])],
        ]);

        $general                          = gs();
        $general->annual_return_rate      = $request->annual_return_rate;
        $general->investment_minimum_year = $request->investment_minimum_year;
        $general->investment_maximum_year = $request->investment_maximum_year;
        $general->penalty_type            = $request->penalty_type;
        $general->contribution_penalty    = $request->contribution_penalty;
        $general->early_closure_penalty   = $request->early_closure_penalty;
        $general->penalty_frequency       = $request->penalty_frequency;
        $general->save();

        $notify[] = ['success', 'Investment setting updated successfully'];
        return back()->withNotify($notify);
    }
}
